/*
 * Copyright (c) 2011, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package javafx.scene.layout;

import com.sun.javafx.util.InterpolationUtils;
import javafx.animation.Interpolatable;
import javafx.beans.NamedArg;
import java.util.Objects;

/**
 * Defines widths for four components (top, right, bottom, and left).
 * Each width is defined as a non-negative
 * value. This value might be interpreted either as an literal value, or as a
 * percentage of the width or height of the Region, depending on the values
 * for {@code topAsPercentage}, {@code rightAsPercentage}, {@code bottomAsPercentage},
 * {@code leftAsPercentage}. The only allowable negative value for top, right,
 * bottom, and left is {@code AUTO}.
 * <p>
 * Because the BorderWidths is immutable, it can safely be used in any
 * cache, and can safely be reused among multiple Regions.
 * @since JavaFX 8.0
 */
public final class BorderWidths implements Interpolatable<BorderWidths> {
    /**
     * When used by a BorderStroke, the value of AUTO is interpreted as the
     * value of {@link BorderStroke#MEDIUM} for the corresponding side. When
     * used with a BorderImage, the value of AUTO means to read the corresponding
     * value from the BorderStroke(s), and not to specify it manually.
     */
    public static final double AUTO = -1;

    /**
     * The default BorderWidths that is used by a BorderImage when null is specified. This
     * width is a single 1 pixel top, right, bottom, and left, all interpreted as literal values.
     */
    public static final BorderWidths DEFAULT = new BorderWidths(1, 1, 1, 1, false, false, false, false);

    /**
     * An empty set of widths, such that all values are 0 and are literal values.
     */
    public static final BorderWidths EMPTY = new BorderWidths(0, 0, 0, 0, false, false, false, false);

    /**
     * A set of widths representing 100% on each side.
     */
    public static final BorderWidths FULL = new BorderWidths(1d, 1d, 1d, 1d, true, true, true, true);


    /**
     * A non-negative value (with the exception of {@link #AUTO}) indicating the border
     * thickness on the top of the border. This value can be a literal value, or can be
     * treated as a percentage, based on the value of the
     * {@link #isTopAsPercentage() topAsPercentage} property.
     *
     * @return the border thickness on the top of the border
     * @interpolationType <a href="../../animation/Interpolatable.html#linear">linear</a> if both values are
     *                    absolute or both values are {@link #isTopAsPercentage() percentages}, and both
     *                    values are not the special value {@link #AUTO};
     *                    <a href="../../animation/Interpolatable.html#discrete">discrete</a> otherwise
     */
    public final double getTop() { return top; }
    private final double top;

    /**
     * The non-negative value (with the exception of {@link #AUTO}) indicating the border
     * thickness on the right of the border. This value can be a literal value, or can be
     * treated as a percentage, based on the value of the
     * {@link #isRightAsPercentage() rightAsPercentage} property.
     *
     * @return the border thickness on the right of the border
     * @interpolationType <a href="../../animation/Interpolatable.html#linear">linear</a> if both values are
     *                    absolute or both values are {@link #isRightAsPercentage() percentages}, and both
     *                    values are not the special value {@link #AUTO};
     *                    <a href="../../animation/Interpolatable.html#discrete">discrete</a> otherwise
     */
    public final double getRight() { return right; }
    private final double right;

    /**
     * The non-negative value (with the exception of {@link #AUTO}) indicating the border
     * thickness on the bottom of the border. This value can be a literal value, or can be
     * treated as a percentage, based on the value of the
     * {@link #isBottomAsPercentage() bottomAsPercentage} property.
     *
     * @return the border thickness on the bottom of the border
     * @interpolationType <a href="../../animation/Interpolatable.html#linear">linear</a> if both values are
     *                    absolute or both values are {@link #isBottomAsPercentage() percentages}, and both
     *                    values are not the special value {@link #AUTO};
     *                    <a href="../../animation/Interpolatable.html#discrete">discrete</a> otherwise
     */
    public final double getBottom() { return bottom; }
    private final double bottom;

    /**
     * The non-negative value (with the exception of {@link #AUTO}) indicating the border
     * thickness on the left of the border. This value can be an literal value, or can be
     * treated as a percentage, based on the value of the
     * {@link #isLeftAsPercentage() leftAsPercentage} property.
     *
     * @return the border thickness on the left of the border
     * @interpolationType <a href="../../animation/Interpolatable.html#linear">linear</a> if both values are
     *                    absolute or both values are {@link #isLeftAsPercentage() percentages}, and both
     *                    values are not the special value {@link #AUTO};
     *                    <a href="../../animation/Interpolatable.html#discrete">discrete</a> otherwise
     */
    public final double getLeft() { return left; }
    private final double left;

    /**
     * Specifies whether the {@link #getTop() top} property should be interpreted as a percentage ({@code true})
     * of the region height or not ({@code false}).
     *
     * @return true if top should be interpreted as a percentage of the region height, otherwise false
     * @interpolationType <a href="../../animation/Interpolatable.html#discrete">discrete</a>
     */
    public final boolean isTopAsPercentage() { return topAsPercentage; }
    private final boolean topAsPercentage;

    /**
     * Specifies whether the {@link #getRight() right} property should be interpreted as a percentage ({@code true})
     * of the region width or not ({@code false}).
     *
     * @return true if right should be interpreted as a percentage of the region width, otherwise false
     * @interpolationType <a href="../../animation/Interpolatable.html#discrete">discrete</a>
     */
    public final boolean isRightAsPercentage() { return rightAsPercentage; }
    private final boolean rightAsPercentage;

    /**
     * Specifies whether the {@link #getBottom() bottom} property should be interpreted as a percentage ({@code true})
     * of the region height or not ({@code false}).
     *
     * @return true if bottom should be interpreted as a percentage of the region height, otherwise false
     * @interpolationType <a href="../../animation/Interpolatable.html#discrete">discrete</a>
     */
    public final boolean isBottomAsPercentage() { return bottomAsPercentage; }
    private final boolean bottomAsPercentage;

    /**
     * Specifies whether the {@link #getLeft() left} property should be interpreted as a percentage ({@code true})
     * of the region width or not ({@code false}).
     *
     * @return true if left should be interpreted as a percentage of the region width, otherwise false
     * @interpolationType <a href="../../animation/Interpolatable.html#discrete">discrete</a>
     */
    public final boolean isLeftAsPercentage() { return leftAsPercentage; }
    private final boolean leftAsPercentage;

    /**
     * A cached hash code for faster secondary usage. It is expected
     * that BorderWidths will be pulled from a cache in many cases.
     */
    private final int hash;

    /**
     * Creates a new BorderWidths using the given width for all four borders,
     * and treating this width as a literal value, and not a percentage.
     *
     * @param width The border width. This cannot be negative.
     */
    public BorderWidths(@NamedArg("width") double width) {
        this(width, width, width, width, false, false, false, false);
    }

    /**
     * Creates a new BorderWidths with the specified widths for top, right,
     * bottom, and left. None of these values may be negative. Each of these
     * values is interpreted as a literal value, not as a percentage.
     *
     * @param top    The thickness of the border on the top. Must be non-negative.
     * @param right    The thickness of the border on the right. Must be non-negative.
     * @param bottom    The thickness of the border on the bottom. Must be non-negative.
     * @param left    The thickness of the border on the left. Must be non-negative.
     */
    public BorderWidths(@NamedArg("top") double top, @NamedArg("right") double right, @NamedArg("bottom") double bottom, @NamedArg("left") double left) {
        this(top, right, bottom, left, false, false, false, false);
    }

    /**
     * Creates a new BorderWidths. None of the values for {@code top}, {@code right}, {@code bottom},
     * or {@code left} can be non-negative.
     *
     * @param top    The thickness of the border on the top. Must be non-negative.
     * @param right    The thickness of the border on the right. Must be non-negative.
     * @param bottom    The thickness of the border on the bottom. Must be non-negative.
     * @param left    The thickness of the border on the left. Must be non-negative.
     * @param topAsPercentage    Whether the top should be treated as a percentage.
     * @param rightAsPercentage    Whether the right should be treated as a percentage.
     * @param bottomAsPercentage    Whether the bottom should be treated as a percentage.
     * @param leftAsPercentage        Whether the left should be treated as a percentage.
     */
    public BorderWidths(
            @NamedArg("top") double top, @NamedArg("right") double right, @NamedArg("bottom") double bottom, @NamedArg("left") double left, @NamedArg("topAsPercentage") boolean topAsPercentage,
            @NamedArg("rightAsPercentage") boolean rightAsPercentage, @NamedArg("bottomAsPercentage") boolean bottomAsPercentage, @NamedArg("leftAsPercentage") boolean leftAsPercentage) {

        // As per CSS 3 Spec (4.3), cannot be negative
        if ((top != AUTO && top < 0) ||
                (right != AUTO && right < 0) ||
                (bottom != AUTO && bottom < 0) ||
                (left != AUTO && left < 0)) {
            throw new IllegalArgumentException("None of the widths can be < 0");
        }

        this.top = top;
        this.right = right;
        this.bottom = bottom;
        this.left = left;
        this.topAsPercentage = topAsPercentage;
        this.rightAsPercentage = rightAsPercentage;
        this.bottomAsPercentage = bottomAsPercentage;
        this.leftAsPercentage = leftAsPercentage;

        // Pre-compute the hash code. NOTE: all variables are prefixed with "this" so that we
        // do not accidentally compute the hash based on the constructor arguments rather than
        // based on the fields themselves!
        int result;
        long temp;
        temp = this.top != +0.0d ? Double.doubleToLongBits(this.top) : 0L;
        result = (int) (temp ^ (temp >>> 32));
        temp = this.right != +0.0d ? Double.doubleToLongBits(this.right) : 0L;
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        temp = this.bottom != +0.0d ? Double.doubleToLongBits(this.bottom) : 0L;
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        temp = this.left != +0.0d ? Double.doubleToLongBits(this.left) : 0L;
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        result = 31 * result + (this.topAsPercentage ? 1 : 0);
        result = 31 * result + (this.rightAsPercentage ? 1 : 0);
        result = 31 * result + (this.bottomAsPercentage ? 1 : 0);
        result = 31 * result + (this.leftAsPercentage ? 1 : 0);
        hash = result;
    }

    /**
     * {@inheritDoc}
     *
     * @throws NullPointerException {@inheritDoc}
     * @since 24
     */
    @Override
    public BorderWidths interpolate(BorderWidths endValue, double t) {
        Objects.requireNonNull(endValue, "endValue cannot be null");

        if (t <= 0 || equals(endValue)) {
            return this;
        }

        if (t >= 1) {
            return endValue;
        }

        double newTop, newRight, newBottom, newLeft;
        boolean newTopAsPercentage, newRightAsPercentage, newBottomAsPercentage, newLeftAsPercentage;

        if (this.topAsPercentage == endValue.topAsPercentage) {
            newTop = interpolate(this.top, endValue.top, t);
            newTopAsPercentage = this.topAsPercentage;
        } else if (t < 0.5) {
            newTop = this.top;
            newTopAsPercentage = this.topAsPercentage;
        } else {
            newTop = endValue.top;
            newTopAsPercentage = endValue.topAsPercentage;
        }

        if (this.rightAsPercentage == endValue.rightAsPercentage) {
            newRight = interpolate(this.right, endValue.right, t);
            newRightAsPercentage = this.rightAsPercentage;
        } else if (t < 0.5) {
            newRight = this.right;
            newRightAsPercentage = this.rightAsPercentage;
        } else {
            newRight = endValue.right;
            newRightAsPercentage = endValue.rightAsPercentage;
        }

        if (this.bottomAsPercentage == endValue.bottomAsPercentage) {
            newBottom = interpolate(this.bottom, endValue.bottom, t);
            newBottomAsPercentage = this.bottomAsPercentage;
        } else if (t < 0.5) {
            newBottom = this.bottom;
            newBottomAsPercentage = this.bottomAsPercentage;
        } else {
            newBottom = endValue.bottom;
            newBottomAsPercentage = endValue.bottomAsPercentage;
        }

        if (this.leftAsPercentage == endValue.leftAsPercentage) {
            newLeft = interpolate(this.left, endValue.left, t);
            newLeftAsPercentage = this.leftAsPercentage;
        } else if (t < 0.5) {
            newLeft = this.left;
            newLeftAsPercentage = this.leftAsPercentage;
        } else {
            newLeft = endValue.left;
            newLeftAsPercentage = endValue.leftAsPercentage;
        }

        if (isSame(newTop, newRight, newBottom, newLeft,
                   newTopAsPercentage, newRightAsPercentage,
                   newBottomAsPercentage, newLeftAsPercentage)) {
            return this;
        }

        if (endValue.isSame(newTop, newRight, newBottom, newLeft,
                            newTopAsPercentage, newRightAsPercentage,
                            newBottomAsPercentage, newLeftAsPercentage)) {
            return endValue;
        }

        return new BorderWidths(
            newTop, newRight, newBottom, newLeft,
            newTopAsPercentage, newRightAsPercentage, newBottomAsPercentage, newLeftAsPercentage);
    }

    private static double interpolate(double start, double end, double t) {
        return start != AUTO && end != AUTO ?
            InterpolationUtils.interpolate(start, end, t) :
            InterpolationUtils.interpolateDiscrete(start, end, t);
    }

    private boolean isSame(double top, double right, double bottom, double left,
                           boolean topAsPercentage, boolean rightAsPercentage,
                           boolean bottomAsPercentage, boolean leftAsPercentage) {
        return this.top == top
            && this.right == right
            && this.bottom == bottom
            && this.left == left
            && this.topAsPercentage == topAsPercentage
            && this.rightAsPercentage == rightAsPercentage
            && this.bottomAsPercentage == bottomAsPercentage
            && this.leftAsPercentage == leftAsPercentage;
    }

    /**
     * {@inheritDoc}
     */
    @Override public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        BorderWidths that = (BorderWidths) o;

        if (this.hash != that.hash) return false;
        if (Double.compare(that.bottom, bottom) != 0) return false;
        if (bottomAsPercentage != that.bottomAsPercentage) return false;
        if (Double.compare(that.left, left) != 0) return false;
        if (leftAsPercentage != that.leftAsPercentage) return false;
        if (Double.compare(that.right, right) != 0) return false;
        if (rightAsPercentage != that.rightAsPercentage) return false;
        if (Double.compare(that.top, top) != 0) return false;
        if (topAsPercentage != that.topAsPercentage) return false;

        return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override public int hashCode() {
        return hash;
    }
}
